<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "fertilizer_prices".
 *
 * @property int $id
 * @property string $district Район
 * @property string $fertilizer Удобрение
 * @property string $typical_price Типичная цена руб/т
 * @property string|null $price_min Минимальная цена руб/т
 * @property string|null $price_max Максимальная цена руб/т
 * @property string|null $price_avg Средняя цена руб/т
 * @property int $created_at Дата создания
 * @property int $updated_at Дата обновления
 */
class FertilizerPrices extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%fertilizer_prices}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['district', 'fertilizer', 'typical_price'], 'required'],
            [['price_min', 'price_max', 'price_avg'], 'number', 'min' => 0],
            [['created_at', 'updated_at'], 'integer'],
            [['district', 'fertilizer'], 'string', 'max' => 100],
            [['typical_price'], 'string', 'max' => 150],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'district' => 'Район',
            'fertilizer' => 'Удобрение',
            'typical_price' => 'Типичная цена (руб/т)',
            'price_min' => 'Минимальная цена (руб/т)',
            'price_max' => 'Максимальная цена (руб/т)',
            'price_avg' => 'Средняя цена (руб/т)',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'timestamp' => [
                'class' => \yii\behaviors\TimestampBehavior::className(),
                'createdAtAttribute' => 'created_at',
                'updatedAtAttribute' => 'updated_at',
            ],
        ];
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            $this->parsePriceRange();
            return true;
        }
        return false;
    }

    protected function parsePriceRange()
    {
        $price = $this->typical_price;

        // Если цена в формате "22000-40000"
        if (strpos($price, '-') !== false) {
            $prices = explode('-', $price);
            $this->price_min = (float)trim($prices[0]);
            $this->price_max = (float)trim($prices[1]);
            $this->price_avg = ($this->price_min + $this->price_max) / 2;
        } else {
            // Если цена фиксированная
            $fixedPrice = (float)$price;
            $this->price_min = $fixedPrice;
            $this->price_max = $fixedPrice;
            $this->price_avg = $fixedPrice;
        }
    }

    /**
     * @param string $district
     * @param string $fertilizer
     * @param string $type min|max|avg (по умолчанию avg)
     * @return float|null
     */
    public static function getPrice($district, $fertilizer, $type = 'avg')
    {
        $model = self::find()
            ->where(['district' => $district, 'fertilizer' => $fertilizer])
            ->one();

        if (!$model) {
            return null;
        }

        switch ($type) {
            case 'min':
                return (float)$model->price_min;
            case 'max':
                return (float)$model->price_max;
            case 'avg':
            default:
                return (float)$model->price_avg;
        }
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getPricesByDistrict($district)
    {
        return self::find()
            ->where(['district' => $district])
            ->indexBy('fertilizer')
            ->all();
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getFertilizersForDistrict($district)
    {
        $models = self::find()
            ->where(['district' => $district])
            ->all();

        $result = [];
        foreach ($models as $model) {
            $result[$model->fertilizer] = $model->fertilizer . ' (' . $model->typical_price . ' руб/т)';
        }

        return $result;
    }

    /**
     * @return array
     */
    public static function getAllFertilizers()
    {
        return self::find()
            ->select('fertilizer')
            ->distinct()
            ->orderBy('fertilizer')
            ->column();
    }

    /**
     * @return array
     */
    public static function getDistrictsList()
    {
        return self::find()
            ->select('district')
            ->distinct()
            ->indexBy('district')
            ->column();
    }

    /**
     * @param string $district
     * @return \yii\db\ActiveQuery
     */
    public static function findByDistrict($district)
    {
        return self::find()->where(['district' => $district]);
    }

    /**
     * @return string
     */
    public function getFormattedPrice()
    {
        if ($this->price_min == $this->price_max) {
            return Yii::$app->formatter->asDecimal($this->price_avg, 0) . ' руб/т';
        } else {
            return Yii::$app->formatter->asDecimal($this->price_min, 0) . '-' .
                Yii::$app->formatter->asDecimal($this->price_max, 0) . ' руб/т';
        }
    }

    /**
     * @param float $amount тонн
     * @param string $type min|max|avg
     * @return float
     */
    public function calculateCost($amount, $type = 'avg')
    {
        $price = $this->getPrice($this->district, $this->fertilizer, $type);
        return $amount * $price;
    }
}