<?php
use yii\helpers\Html;
use yii\helpers\Url;

/** @var yii\web\View $this */
/** @var app\models\Field[] $fields */
/** @var app\models\Field|null $selectedField */

$this->title = 'Калькулятор удобрений с ИИ';

// Получаем ВСЕ данные из БД для передачи в ИИ
$allNorms = \app\models\FertilizerNorms::find()->all();
$allFertilizerPrices = \app\models\FertilizerPrices::find()->all();
$allProductPrices = \app\models\ProductPrices::find()->all();
$allCropRotations = \app\models\CropRotation::find()->all();

// Данные для выбранного поля
$selectedFieldNorms = null;
$fertilizerPrices = null;
$productPrice = null;
$cropRotations = [];

if ($selectedField) {
    $selectedFieldNorms = \app\models\FertilizerNorms::getNorms($selectedField->district, $selectedField->desired_culture);
    $fertilizerPrices = \app\models\FertilizerPrices::getPricesByDistrict($selectedField->district);
    $productPrice = \app\models\ProductPrices::getPrice($selectedField->district, $selectedField->desired_culture);
    $cropRotations = \app\models\CropRotation::findByDistrict($selectedField->district)->limit(2)->all();
}
?>

<div class="container-fluid">
    <div class="row">
        <main class="col-md-12">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2 text-success">
                    <i class="bi bi-calculator me-2"></i> Калькулятор удобрений
                </h1>
                <div>
                    <a href="<?= Url::to(['giga-chat/fields']) ?>" class="btn btn-outline-success btn-sm me-2">
                        <i class="bi bi-map me-1"></i> Мои поля
                    </a>
                    <a href="<?= Url::to(['giga-chat/field']) ?>" class="btn btn-success btn-sm">
                        <i class="bi bi-plus-circle me-1"></i> Добавить поле
                    </a>
                </div>
            </div>

            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-info-circle me-2"></i> ИИ проанализирует данные поля для точных расчетов удобрений и прибыльности
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>

            <div class="row">
                <div class="col-lg-4">
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 bg-success text-white">
                            <h6 class="m-0 fw-bold">
                                <i class="bi bi-map me-2"></i> Выберите поле
                            </h6>
                        </div>
                        <div class="card-body" style="max-height: 500px; overflow-y: auto;">
                            <?php if (empty($fields)): ?>
                                <div class="text-center py-4">
                                    <i class="bi bi-map text-muted" style="font-size: 3rem;"></i>
                                    <p class="mt-3 text-muted">Поля не добавлены</p>
                                    <a href="<?= Url::to(['giga-chat/field']) ?>" class="btn btn-success btn-sm">
                                        <i class="bi bi-plus-circle me-1"></i> Добавить первое поле
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="list-group">
                                    <?php foreach ($fields as $field): ?>
                                        <a href="<?= Url::to(['giga-chat/calc', 'field_id' => $field->id]) ?>"
                                           class="list-group-item list-group-item-action field-item <?= $selectedField && $selectedField->id == $field->id ? 'active' : '' ?>">
                                            <div class="d-flex w-100 justify-content-between">
                                                <h6 class="mb-1"><?= Html::encode($field->name) ?></h6>
                                                <small><?= Yii::$app->formatter->asDecimal($field->area) ?> га</small>
                                            </div>
                                            <p class="mb-1">
                                                <span class="badge bg-secondary"><?= Html::encode($field->district) ?></span>
                                                <span class="badge bg-success"><?= Html::encode($field->desired_culture) ?></span>
                                            </p>
                                            <small class="text-muted">
                                                <?= Html::encode($field->soil_type) ?>
                                                <?php if ($field->predecessor): ?>
                                                    • Предшественник: <?= Html::encode($field->predecessor) ?>
                                                <?php endif; ?>
                                            </small>
                                        </a>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <div class="col-lg-8">
                    <?php if ($selectedField): ?>
                        <div class="card shadow mb-4">
                            <div class="card-header py-3 bg-success text-white">
                                <h6 class="m-0 fw-bold">
                                    <i class="bi bi-geo-alt me-2"></i> <?= Html::encode($selectedField->name) ?>
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <table class="table table-sm table-borderless">
                                            <tr>
                                                <td width="40%"><strong>Район:</strong></td>
                                                <td><?= Html::encode($selectedField->district) ?></td>
                                            </tr>
                                            <tr>
                                                <td><strong>Культура:</strong></td>
                                                <td class="text-success fw-bold"><?= Html::encode($selectedField->desired_culture) ?></td>
                                            </tr>
                                            <tr>
                                                <td><strong>Площадь:</strong></td>
                                                <td><?= Yii::$app->formatter->asDecimal($selectedField->area) ?> га</td>
                                            </tr>
                                            <?php if ($selectedField->nitrogen_content): ?>
                                                <tr>
                                                    <td><strong>Азот:</strong></td>
                                                    <td><?= Yii::$app->formatter->asDecimal($selectedField->nitrogen_content) ?> мг/кг</td>
                                                </tr>
                                            <?php endif; ?>
                                            <?php if ($selectedField->phosphorus_content): ?>
                                                <tr>
                                                    <td><strong>Фосфор:</strong></td>
                                                    <td><?= Yii::$app->formatter->asDecimal($selectedField->phosphorus_content) ?> мг/кг</td>
                                                </tr>
                                            <?php endif; ?>
                                            <?php if ($selectedField->potassium_content): ?>
                                                <tr>
                                                    <td><strong>Калий:</strong></td>
                                                    <td><?= Yii::$app->formatter->asDecimal($selectedField->potassium_content) ?> мг/кг</td>
                                                </tr>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                    <div class="col-md-6">
                                        <table class="table table-sm table-borderless">
                                            <tr>
                                                <td width="40%"><strong>Тип почвы:</strong></td>
                                                <td><?= Html::encode($selectedField->soil_type) ?></td>
                                            </tr>
                                            <?php if ($selectedField->predecessor): ?>
                                                <tr>
                                                    <td><strong>Предшественник:</strong></td>
                                                    <td><?= Html::encode($selectedField->predecessor) ?></td>
                                                </tr>
                                            <?php endif; ?>
                                            <?php if ($selectedField->humus_content): ?>
                                                <tr>
                                                    <td><strong>Гумус:</strong></td>
                                                    <td><?= Yii::$app->formatter->asDecimal($selectedField->humus_content) ?> %</td>
                                                </tr>
                                            <?php endif; ?>
                                            <?php if ($selectedField->ph_level): ?>
                                                <tr>
                                                    <td><strong>pH почвы:</strong></td>
                                                    <td><?= Yii::$app->formatter->asDecimal($selectedField->ph_level) ?></td>
                                                </tr>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                </div>

                                <?php
                                $norms = $selectedField->getRecommendedNorms();
                                $yieldForecast = $selectedField->getYieldForecast();
                                $costCalculation = $selectedField->calculateFertilizerCost();
                                ?>
                            </div>
                        </div>

                        <div class="card shadow">
                            <div class="card-header py-3 bg-success text-white">
                                <h6 class="m-0 fw-bold">
                                    <i class="bi bi-robot me-2"></i> Анализ и рекомендации ИИ
                                </h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <label class="form-label">Дополнительные параметры (опционально)</label>
                                    <textarea class="form-control" rows="3"
                                              placeholder="Укажите дополнительные условия: ожидаемая урожайность, бюджет на удобрения, особенности агротехники и т.д."
                                              id="additional-params"></textarea>
                                </div>

                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted">
                                        ИИ проанализирует данные поля и базу данных для точных расчетов
                                    </small>
                                    <button type="button" class="btn btn-success" id="calculate-btn">
                                        <i class="bi bi-calculator me-1"></i> Запустить анализ
                                    </button>
                                </div>

                                <div id="loading" style="display: none; margin-top: 15px;">
                                    <div class="text-center">
                                        <div class="spinner-border text-success" role="status">
                                            <span class="visually-hidden">Загрузка...</span>
                                        </div>
                                        <p class="mt-2 text-muted">ИИ анализирует данные поля...</p>
                                    </div>
                                </div>

                                <div id="ai-response" style="display: none; margin-top: 20px;"></div>
                            </div>
                        </div>

                    <?php else: ?>
                        <div class="card shadow">
                            <div class="card-body text-center py-5">
                                <i class="bi bi-calculator text-muted" style="font-size: 4rem;"></i>
                                <h3 class="text-muted mt-3">Выберите поле для анализа</h3>
                                <p class="text-muted">Выберите поле из списка слева для получения рекомендаций по удобрениям</p>
                                <?php if (empty($fields)): ?>
                                    <a href="<?= Url::to(['giga-chat/field']) ?>" class="btn btn-success">
                                        <i class="bi bi-plus-circle me-1"></i> Добавить первое поле
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

        </main>
    </div>
</div>

<input type="hidden" id="session-id" value="<?= $sessionId ?>">
<input type="hidden" id="selected-field-id" value="<?= $selectedField ? $selectedField->id : '' ?>">

<!-- ВЫВОД ДАННЫХ ИЗ БД -->
<div id="db-data" style="display: none;"
     data-selected-field-norms='<?= $selectedFieldNorms ? json_encode($selectedFieldNorms->getNormsArray()) : "null" ?>'
     data-selected-field-norms-basis='<?= $selectedFieldNorms ? Html::encode($selectedFieldNorms->recommendation_basis) : "Нормы не найдены" ?>'
     data-all-norms='<?= json_encode(array_map(function($norm) {
         return [
             'district' => $norm->district,
             'culture' => $norm->culture,
             'n_kg_ha' => $norm->n_kg_ha,
             'p2o5_kg_ha' => $norm->p2o5_kg_ha,
             'k2o_kg_ha' => $norm->k2o_kg_ha,
             'recommendation_basis' => $norm->recommendation_basis
         ];
     }, $allNorms)) ?>'
     data-all-fertilizer-prices='<?= json_encode(array_map(function($price) {
         return [
             'district' => $price->district,
             'fertilizer' => $price->fertilizer,
             'typical_price' => $price->typical_price,
             'price_avg' => $price->price_avg
         ];
     }, $allFertilizerPrices)) ?>'
     data-all-product-prices='<?= json_encode(array_map(function($price) {
         return [
             'district' => $price->district,
             'product' => $price->product,
             'price' => $price->price,
             'notes' => $price->notes
         ];
     }, $allProductPrices)) ?>'
     data-all-crop-rotations='<?= json_encode(array_map(function($rotation) {
         return [
             'district' => $rotation->district,
             'rotation_name' => $rotation->rotation_name,
             'year_1' => $rotation->year_1,
             'year_2' => $rotation->year_2,
             'year_3' => $rotation->year_3,
             'year_4' => $rotation->year_4,
             'notes' => $rotation->notes
         ];
     }, $allCropRotations)) ?>'>
</div>

<style>
    .field-item:hover { transform: translateX(2px); transition: transform 0.2s ease; }
    .field-item.active { background-color: #198754 !important; border-color: #198754 !important; }

    .ai-report-container {
        background: linear-gradient(135deg, #f8fff8 0%, #f0f8f0 100%);
        border: 1px solid #d1e7dd;
        border-radius: 12px;
        padding: 2rem;
        margin: 1rem 0;
        box-shadow: 0 4px 6px rgba(25, 135, 84, 0.1);
    }

    .ai-section {
        background: white;
        border-radius: 8px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
        border-left: 4px solid #198754;
        box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    }

    .ai-section-title {
        color: #198754;
        font-weight: 600;
        font-size: 1.3rem;
        margin-bottom: 1rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #e9f7ef;
    }

    .ai-subsection {
        background: #f8f9fa;
        border-radius: 6px;
        padding: 1rem;
        margin: 1rem 0;
        border-left: 3px solid #20c997;
    }

    .ai-subtitle {
        color: #198754;
        font-weight: 600;
        font-size: 1.1rem;
        margin-bottom: 0.75rem;
    }

    .ai-table {
        width: 100%;
        border-collapse: collapse;
        margin: 1rem 0;
        font-size: 0.9rem;
        background: white;
        border-radius: 8px;
        overflow: hidden;
        box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    }

    .ai-table th {
        background: linear-gradient(135deg, #198754 0%, #157347 100%);
        color: white;
        padding: 1rem;
        text-align: left;
        font-weight: 600;
        border: none;
    }

    .ai-table td {
        padding: 0.75rem 1rem;
        border-bottom: 1px solid #e9ecef;
        vertical-align: middle;
    }

    .ai-table tr:hover {
        background-color: #f8f9fa;
    }

    .ai-table .text-end { text-align: right; }
    .ai-table .positive { color: #198754; font-weight: 600; }
    .ai-table .negative { color: #dc3545; }

    .calculation-block {
        background: linear-gradient(135deg, #e8f5e8 0%, #d4edda 100%);
        border: 1px solid #c3e6cb;
        border-radius: 8px;
        padding: 1.25rem;
        margin: 1rem 0;
    }

    .calculation-formula {
        font-family: 'Courier New', monospace;
        font-size: 1rem;
        background: rgba(255,255,255,0.7);
        padding: 0.75rem 1rem;
        border-radius: 6px;
        margin: 0.5rem 0;
        border-left: 3px solid #198754;
    }

    .calculation-result {
        font-weight: 700;
        color: #198754;
        background: rgba(25, 135, 84, 0.1);
        padding: 0.2rem 0.5rem;
        border-radius: 4px;
    }

    .math-equation {
        background: #f8f9fa;
        padding: 1rem;
        border-radius: 6px;
        margin: 0.75rem 0;
        font-family: 'Courier New', monospace;
        border-left: 3px solid #20c997;
    }

    /* Списки */
    .ai-list {
        margin: 0.5rem 0;
        padding-left: 1.5rem;
    }

    .ai-list li {
        margin-bottom: 0.5rem;
        line-height: 1.5;
        padding-left: 0.5rem;
    }

    .ai-list li::marker {
        color: #198754;
    }

    .recommendation-block {
        background: linear-gradient(135deg, #d1e7dd 0%, #c3e6cb 100%);
        border: 1px solid #b7dac4;
        border-radius: 8px;
        padding: 1.5rem;
        margin: 1.5rem 0;
    }

    .recommendation-title {
        color: #0f5132;
        font-weight: 700;
        font-size: 1.1rem;
        margin-bottom: 1rem;
        display: flex;
        align-items: center;
    }

    .recommendation-title::before {
        content: "💡";
        margin-right: 0.5rem;
        font-size: 1.2rem;
    }

    .ai-h1 {
        color: #0f5132;
        font-size: 1.8rem;
        font-weight: 700;
        margin: 2rem 0 1.5rem 0;
        padding-bottom: 0.75rem;
        border-bottom: 3px solid #198754;
    }

    .ai-h2 {
        color: #198754;
        font-size: 1.5rem;
        font-weight: 600;
        margin: 1.8rem 0 1.2rem 0;
        padding-left: 0.5rem;
        border-left: 4px solid #198754;
    }

    .ai-h3 {
        color: #20c997;
        font-size: 1.2rem;
        font-weight: 600;
        margin: 1.5rem 0 1rem 0;
    }

    .status-indicator {
        display: inline-block;
        padding: 0.3rem 0.8rem;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
        margin: 0.2rem;
    }

    .status-good {
        background: linear-gradient(135deg, #d1e7dd 0%, #a3cfbb 100%);
        color: #0f5132;
        border: 1px solid #75b798;
    }

    .status-warning {
        background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
        color: #856404;
        border: 1px solid #ffda6a;
    }

    .status-bad {
        background: linear-gradient(135deg, #f8d7da 0%, #f1aeb5 100%);
        color: #721c24;
        border: 1px solid #e6858f;
    }

    .ai-paragraph {
        line-height: 1.7;
        color: #495057;
        margin-bottom: 1.2rem;
        text-align: justify;
    }

    @media (max-width: 768px) {
        .ai-report-container {
            padding: 1rem;
            margin: 0.5rem 0;
        }

        .ai-section {
            padding: 1rem;
        }

        .ai-table {
            font-size: 0.8rem;
        }

        .ai-table th,
        .ai-table td {
            padding: 0.5rem;
        }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const calculateBtn = document.getElementById('calculate-btn');
        if (calculateBtn) {
            calculateBtn.addEventListener('click', function(e) {
                e.preventDefault();
                sendAIRequest();
            });
        }
    });

    function formatChatResponse(response) {
        if (!response) return '<div class="text-muted text-center p-4">Нет данных для отображения</div>';

        let html = response;

        // 1. Тщательная очистка текста от всех лишних символов
        html = cleanText(html);

        // 2. Обработка заголовков
        html = formatHeadings(html);

        // 3. Обработка таблиц
        html = formatTables(html);

        // 4. Обработка списков
        html = formatLists(html);

        // 5. Обработка формул и расчетов
        html = formatCalculations(html);

        // 6. Форматирование абзацев
        html = formatParagraphs(html);

        // 7. Финальная очистка
        html = finalCleanup(html);

        return `<div class="ai-report-container">${html}</div>`;
    }

    function cleanText(html) {
        html = html.replace(/\\text\{([^}]*)\}/g, '$1');
        html = html.replace(/\\frac\{([^}]*)\}\{([^}]*)\}/g, '($1)/($2)');
        html = html.replace(/\\approx/g, '≈');
        html = html.replace(/\\times/g, '×');
        html = html.replace(/\\quad/g, ' ');
        html = html.replace(/\\\w+/g, '');
        html = html.replace(/\$\$([^$]*)\$\$/g, '$1');
        html = html.replace(/\\\(([^)]*)\\\)/g, '$1');
        html = html.replace(/\{\}/g, '');
        html = html.replace(/\$([^$\s]+)\$/g, '$1');
        html = html.replace(/(\d)\s*\n\s*([×\-\+])/g, '$1 $2');
        html = html.replace(/([×\-\+])\s*\n\s*(\d)/g, '$1 $2');
        html = html.replace(/\*\*([^*]+)\*/g, '**$1**');
        html = html.replace(/\*([^*]+)\*\*/g, '**$1**');

        return html;
    }

    function formatHeadings(html) {
        return html
            .replace(/^# (.+)$/gm, '<div class="ai-h1">$1</div>')
            .replace(/^## (.+)$/gm, '<div class="ai-h2">$1</div>')
            .replace(/^### (.+)$/gm, '<div class="ai-h3">$1</div>')
            .replace(/^(\d+)\.\s+(.+)$/gm, '<div class="ai-h2">$1. $2</div>')
            .replace(/^(\d+\.\d+)\.\s+(.+)$/gm, '<div class="ai-h3">$1. $2</div>');
    }

    function formatTables(html) {
        const tableRegex = /(\|.+\|(?:\n\|.+\|)+)/g;

        return html.replace(tableRegex, function(match) {
            const rows = match.trim().split('\n').filter(row => row.trim());

            let tableHtml = '<div class="table-responsive mt-4"><table class="ai-table">';

            rows.forEach((row, index) => {
                const cells = row.split('|').filter(cell => cell.trim() !== '');

                if (index === 0) {
                    tableHtml += '<thead><tr>';
                    cells.forEach(cell => {
                        tableHtml += `<th>${cell.trim()}</th>`;
                    });
                    tableHtml += '</tr></thead><tbody>';
                } else if (cells.some(cell => cell.includes('---'))) {
                    return;
                } else {
                    tableHtml += '<tr>';
                    cells.forEach((cell) => {
                        const cellContent = cell.trim();
                        const isNumeric = /^-?\d+([.,]\d+)?%?$/.test(cellContent) ||
                            cellContent.includes('руб') ||
                            cellContent.includes('ц/га');

                        const cellClass = isNumeric ? 'text-end' : '';
                        tableHtml += `<td class="${cellClass}">${cellContent}</td>`;
                    });
                    tableHtml += '</tr>';
                }
            });

            tableHtml += '</tbody></table></div>';
            return tableHtml;
        });
    }

    function formatLists(html) {
        html = html.replace(/(\d+\.\s+.+?)(?=\n\d+\.|\n##|\n#|\n$)/gs, function(match) {
            const items = match.split('\n').filter(item => item.trim());
            let listHtml = '<ol class="ai-list">';

            items.forEach(item => {
                const content = item.replace(/^\d+\.\s*/, '').trim();
                if (content) {
                    listHtml += `<li>${content}</li>`;
                }
            });

            listHtml += '</ol>';
            return listHtml;
        });

        html = html.replace(/([-•*]\s+.+?)(?=\n[-•*]|\n##|\n#|\n$)/gs, function(match) {
            const items = match.split('\n').filter(item => item.trim());
            let listHtml = '<ul class="ai-list">';

            items.forEach(item => {
                const content = item.replace(/^[-•*]\s*/, '').trim();
                if (content) {
                    listHtml += `<li>${content}</li>`;
                }
            });

            listHtml += '</ul>';
            return listHtml;
        });

        return html;
    }

    function formatCalculations(html) {
        html = html.replace(/(Дефицит\s+\w+\s*=\s*)([^=]+)(=\s*[^%\n]+%)/g,
            '<div class="calculation-block">$1$2$3</div>');
        html = html.replace(/(\d+\.?\d*)\s*×\s*(\d+\.?\d*)\s*×\s*(\d+\.?\d*)\s*=\s*([\d\s,.]+)/g,
            '<div class="calculation-simple">$1 × $2 × $3 = <strong>$4</strong></div>');
        html = html.replace(/(\d+\.?\d*)\s*×\s*(\d+\.?\d*)\s*=\s*([\d\s,.]+)/g,
            '<div class="calculation-simple">$1 × $2 = <strong>$3</strong></div>');
        html = html.replace(/(\d+)\s*-\s*(\d+)\s*=\s*([\d.-]+)/g,
            '<div class="calculation-simple">$1 - $2 = <strong>$3</strong></div>');
        html = html.replace(/(-?\d+\.?\d*%)/g, '<span class="calculation-result">$1</span>');
        html = html.replace(/(\d+[\d\s,.]*руб\/га)/gi, '<span class="calculation-money">$1</span>');
        html = html.replace(/(\d+[\d\s,.]*руб)/gi, '<span class="calculation-money">$1</span>');
        html = html.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');

        return html;
    }

    function formatParagraphs(html) {
        const lines = html.split('\n');
        let cleanedLines = [];

        let currentParagraph = '';

        lines.forEach(line => {
            const trimmedLine = line.trim();

            if (!trimmedLine) {
                if (currentParagraph) {
                    cleanedLines.push(`<p class="ai-paragraph">${currentParagraph}</p>`);
                    currentParagraph = '';
                }
                return;
            }

            if (trimmedLine.match(/^<div class="ai-h[1-3]"/) ||
                trimmedLine.match(/^<div class="table-responsive"/) ||
                trimmedLine.match(/^<ol class="ai-list"/) ||
                trimmedLine.match(/^<ul class="ai-list"/) ||
                trimmedLine.match(/^<div class="calculation-simple"/) ||
                trimmedLine.match(/^<div class="calculation-block"/)) {

                if (currentParagraph) {
                    cleanedLines.push(`<p class="ai-paragraph">${currentParagraph}</p>`);
                    currentParagraph = '';
                }
                cleanedLines.push(trimmedLine);
            } else if (trimmedLine.match(/^\d+\.\s/) || trimmedLine.match(/^[-•*]\s/)) {
                if (currentParagraph) {
                    cleanedLines.push(`<p class="ai-paragraph">${currentParagraph}</p>`);
                    currentParagraph = '';
                }
            } else {
                currentParagraph += (currentParagraph ? ' ' : '') + trimmedLine;
            }
        });

        if (currentParagraph) {
            cleanedLines.push(`<p class="ai-paragraph">${currentParagraph}</p>`);
        }

        return cleanedLines.join('\n');
    }

    function finalCleanup(html) {
        html = html.replace(/\$/g, '');

        html = html.replace(/\n{3,}/g, '\n\n');

        html = html.replace(/<strong>([^<]*)<\/strong>\*/g, '<strong>$1</strong>');
        html = html.replace(/\*<strong>([^<]*)<\/strong>/g, '<strong>$1</strong>');

        return html;
    }

    const improvedStyles = `
<style>
    .calculation-block {
        background: #f8f9fa;
        padding: 1rem;
        border-radius: 8px;
        margin: 1rem 0;
        border-left: 4px solid #198754;
        font-family: 'Courier New', monospace;
    }

    .calculation-simple {
        background: #e8f5e8;
        padding: 0.75rem;
        border-radius: 6px;
        margin: 0.5rem 0;
        border-left: 3px solid #28a745;
    }

    .calculation-result {
        color: #198754;
        font-weight: bold;
        background: rgba(25, 135, 84, 0.1);
        padding: 0.2rem 0.4rem;
        border-radius: 4px;
    }

    .calculation-money {
        color: #dc3545;
        font-weight: bold;
        background: rgba(220, 53, 69, 0.1);
        padding: 0.2rem 0.4rem;
        border-radius: 4px;
    }

    .ai-list li {
        margin-bottom: 0.5rem;
        line-height: 1.5;
        padding-left: 0.5rem;
    }

    .ai-paragraph {
        line-height: 1.6;
        margin-bottom: 1rem;
        text-align: justify;
    }

    strong {
        color: #2c5530;
        font-weight: 600;
    }
</style>
`;

    document.head.insertAdjacentHTML('beforeend', improvedStyles);

    document.head.insertAdjacentHTML('beforeend', simpleStyles);

    function sendAIRequest() {
        console.log('Запуск анализа с ИИ...');

        const fieldId = document.getElementById('selected-field-id').value;
        const additionalParams = document.getElementById('additional-params').value;

        if (!fieldId) {
            alert('Пожалуйста, выберите поле для анализа');
            return;
        }

        const loading = document.getElementById('loading');
        const aiResponse = document.getElementById('ai-response');
        const calculateBtn = document.getElementById('calculate-btn');

        loading.style.display = 'block';
        aiResponse.style.display = 'none';
        calculateBtn.disabled = true;

        const dbData = document.getElementById('db-data');
        if (!dbData) {
            alert('Ошибка загрузки данных');
            loading.style.display = 'none';
            calculateBtn.disabled = false;
            return;
        }

        const selectedFieldNorms = JSON.parse(dbData.dataset.selectedFieldNorms || 'null');
        const selectedFieldNormsBasis = dbData.dataset.selectedFieldNormsBasis;
        const allNorms = JSON.parse(dbData.dataset.allNorms || '[]');
        const allFertilizerPrices = JSON.parse(dbData.dataset.allFertilizerPrices || '[]');
        const allProductPrices = JSON.parse(dbData.dataset.allProductPrices || '[]');
        const allCropRotations = JSON.parse(dbData.dataset.allCropRotations || '[]');

        const activeField = document.querySelector('.field-item.active');
        const fieldData = {
            name: activeField.querySelector('h6').textContent.trim(),
            district: activeField.querySelector('.badge.bg-secondary').textContent.trim(),
            culture: activeField.querySelector('.badge.bg-success').textContent.trim(),
            area: parseFloat(activeField.querySelector('small').textContent.replace(' га', '').trim())
        };

        const tables = document.querySelectorAll('.table-borderless');
        let soil = '', ph = 'Не указан', predecessor = 'Не указан';
        let nitrogenContent = 'Не указано', phosphorusContent = 'Не указано',
            potassiumContent = 'Не указано', humusContent = 'Не указано';

        tables.forEach(table => {
            const rows = table.querySelectorAll('tr');
            rows.forEach(row => {
                const cells = row.querySelectorAll('td');
                if (cells.length === 2) {
                    const key = cells[0].textContent.trim();
                    const value = cells[1].textContent.trim();

                    if (key.includes('Тип почвы')) soil = value;
                    else if (key.includes('pH почвы')) ph = value;
                    else if (key.includes('Предшественник')) predecessor = value;
                    else if (key.includes('Азот')) nitrogenContent = value.replace(' мг/кг', '');
                    else if (key.includes('Фосфор')) phosphorusContent = value.replace(' мг/кг', '');
                    else if (key.includes('Калий')) potassiumContent = value.replace(' мг/кг', '');
                    else if (key.includes('Гумус')) humusContent = value.replace(' %', '');
                }
            });
        });

        let message = `Ты — агроном-экономист. Проанализируй данные поля и сделай точные расчеты по культурам на основе ВСЕХ доступных данных из базы.

=== ДАННЫЕ ПОЛЯ ===
• Название: ${fieldData.name}
• Район: ${fieldData.district}
• Текущая культура: ${fieldData.culture}
• Площадь: ${fieldData.area} га
• Тип почвы: ${soil}
• Уровень pH: ${ph}
• Предшественник: ${predecessor}

=== АНАЛИЗ ПОЧВЫ ===
• Азот (N): ${nitrogenContent} мг/кг
• Фосфор (P): ${phosphorusContent} мг/кг
• Калий (K): ${potassiumContent} мг/кг
• Гумус: ${humusContent}%

=== БАЗА ДАННЫХ: ВСЕ НОРМЫ УДОБРЕНИЙ ===
${allNorms.length > 0 ? allNorms.map(norm =>
            `• ${norm.district} - ${norm.culture}: N=${norm.n_kg_ha} кг/га, P₂O₅=${norm.p2o5_kg_ha} кг/га, K₂O=${norm.k2o_kg_ha} кг/га (${norm.recommendation_basis})`
        ).join('\n') : '• Нормы не найдены в базе данных'}

=== БАЗА ДАННЫХ: ВСЕ ЦЕНЫ НА УДОБРЕНИЯ ===
${allFertilizerPrices.length > 0 ? allFertilizerPrices.map(fp =>
            `• ${fp.district} - ${fp.fertilizer}: ${fp.typical_price} руб/т`
        ).join('\n') : '• Данные о ценах не найдены'}

=== БАЗА ДАННЫХ: ВСЕ ЦЕНЫ НА ПРОДУКЦИЮ ===
${allProductPrices.length > 0 ? allProductPrices.map(pp =>
            `• ${pp.district} - ${pp.product}: ${pp.price} руб/т${pp.notes ? ` (${pp.notes})` : ''}`
        ).join('\n') : '• Цены на продукцию не найдены'}

=== БАЗА ДАННЫХ: ВСЕ СЕВООБОРОТЫ ===
${allCropRotations.length > 0 ? allCropRotations.map(cr =>
            `• ${cr.district} - ${cr.rotation_name}: ${cr.year_1} → ${cr.year_2} → ${cr.year_3} → ${cr.year_4}${cr.notes ? ` (${cr.notes})` : ''}`
        ).join('\n') : '• Данные о севообороте не найдены'}

=== НОРМЫ ДЛЯ ТЕКУЩЕЙ КУЛЬТУРЫ ===
${selectedFieldNorms ?
            `• Азот (N): ${selectedFieldNorms.n} кг/га
• Фосфор (P₂O₅): ${selectedFieldNorms.p2o5} кг/га
• Калий (K₂O): ${selectedFieldNorms.k2o} кг/га
• Основание: ${selectedFieldNormsBasis}` :
            '• Нормы для текущей культуры не найдены'}

=== ДОПОЛНИТЕЛЬНЫЕ ТРЕБОВАНИЯ ===
${additionalParams || 'Не указаны'}

=== ЗАДАЧА ===
На основе ВСЕХ данных из базы рассчитай для 5-7 культур:

1. РАСЧЕТ ДЕФИЦИТА УДОБРЕНИЙ для каждой культуры
2. РАСЧЕТ СТОИМОСТИ УДОБРЕНИЙ с использованием актуальных цен
3. РАСЧЕТ ПРИБЫЛЬНОСТИ с учетом цен на продукцию
4. УЧЕТ СЕВООБОРОТА

=== ТРЕБОВАНИЯ К ОТВЕТУ ===
1. **ОБЯЗАТЕЛЬНО создай сравнительную таблицу культур** в точном формате:
| Культура | Урожайность (ц/га) | Цена (руб/т) | Выручка (руб/га) | Затраты на удобрения (руб/га) | ПРИБЫЛЬ (РУБ/ГА) |
|----------|-------------------|-------------|------------------|-----------------------------|------------------|
2. **Столбец "ПРИБЫЛЬ (РУБ/ГА)" должен быть ПОСЛЕДНИМ** и содержать ТОЛЬКО ЦИФРЫ

3. **Укажи явно самую выгодную культуру** в тексте:
"Самая выгодная культура — ПШЕНИЦА (395 000 руб/га)"

4. **Используй только реалистичные цифры прибыли**: от 10 000 до 500 000 руб/га

5. **Форматируй числа с пробелами**: 395 000 (не 395000)

=== ЧТО НЕ ДЕЛАТЬ ===
- НЕ меняй порядок столбцов в таблице
- НЕ объединяй ячейки в таблице
- НЕ используй формулы в ячейках (только готовые числа)
- НЕ пиши прибыль в других валютах кроме руб/га

• Используй точные математические расчеты
• Покажи формулы типа: 74.5 × 10 × 50 = 37250 руб/га
• Создай сравнительную таблицу культур
• Дай детальные расчеты для топ-3 культур
• Предложи рекомендации по севообороту


Отвечай на русском языке с профессиональными расчетами.`;

        fetch('<?= Url::to(['giga-chat/send-message']) ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-Token': '<?= Yii::$app->request->csrfToken ?>'
            },
            body: new URLSearchParams({
                'message': message,
                '_csrf': '<?= Yii::$app->request->csrfToken ?>'
            })
        })
            .then(response => response.json())
            .then(data => {
                loading.style.display = 'none';
                calculateBtn.disabled = false;

                if (data.success) {
                    aiResponse.innerHTML =
                        '<div class="alert alert-success">' +
                        '<h5><i class="bi bi-check-circle me-2"></i>Расчеты готовы</h5>' +
                        '<div class="mt-3">' + formatChatResponse(data.response) + '</div>' +
                        '</div>';
                    aiResponse.style.display = 'block';
                    analyzeAndSaveCalculation(data.response, fieldId, fieldData.area);

                } else {
                    aiResponse.innerHTML =
                        '<div class="alert alert-danger">' +
                        '<i class="bi bi-exclamation-triangle me-2"></i>Ошибка: ' + (data.error || 'Неизвестная ошибка') +
                        '</div>';
                    aiResponse.style.display = 'block';
                }
            })
            .catch(error => {
                console.error('Ошибка запроса:', error);
                loading.style.display = 'none';
                calculateBtn.disabled = false;
                aiResponse.innerHTML =
                    '<div class="alert alert-danger">' +
                    '<i class="bi bi-exclamation-triangle me-2"></i>Ошибка: ' + error.message +
                    '</div>';
                aiResponse.style.display = 'block';
            });
    }

    function analyzeAndSaveCalculation(aiResponse, fieldId, area) {
        try {
            console.log('Начало анализа ответа ИИ...');

            const profitData = parseProfitData(aiResponse);

            console.log('Результаты анализа:', {
                bestCrop: profitData.bestCrop,
                profitPerHa: profitData.profitPerHa,
                totalCrops: profitData.crops.length,
                crops: profitData.crops
            });

            if (profitData.bestCrop && profitData.profitPerHa > 0) {
                saveCalculationToDatabase(fieldId, profitData, area, aiResponse);
            } else {
                console.warn('Не удалось определить самую прибыльную культуру');
                const notification = document.createElement('div');
                notification.className = 'alert alert-warning alert-dismissible fade show';
                notification.innerHTML = `
                <i class="bi bi-exclamation-triangle me-2"></i>
                <strong>Внимание!</strong> Не удалось автоматически определить самую прибыльную культуру.
                Пожалуйста, проверьте рекомендации ИИ вручную.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
                document.getElementById('ai-response').prepend(notification);
            }
        } catch (error) {
            console.error('Ошибка анализа данных:', error);

            const errorNotification = document.createElement('div');
            errorNotification.className = 'alert alert-danger alert-dismissible fade show';
            errorNotification.innerHTML = `
            <i class="bi bi-bug me-2"></i>
            <strong>Ошибка анализа!</strong> ${error.message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
            document.getElementById('ai-response').prepend(errorNotification);
        }
    }
    function parseProfitData(aiResponse) {
        const result = {
            bestCrop: null,
            profitPerHa: 0,
            crops: []
        };

        console.log('Начинаем поиск таблицы в ответе ИИ...');
        const tableRegex = /(\|.*\|(?:\n\|.*\|)+)/g;
        const tables = aiResponse.match(tableRegex);

        if (!tables || tables.length === 0) {
            console.warn('Таблицы не найдены в ответе ИИ');
            return result;
        }

        console.log(`Найдено таблиц: ${tables.length}`);
        tables.forEach((table, tableIndex) => {
            console.log(`Анализ таблицы ${tableIndex + 1}:`);

            const rows = table.trim().split('\n').filter(row => row.trim() && !row.includes('---'));

            if (rows.length < 2) {
                console.log('Таблица слишком мала, пропускаем');
                return;
            }

            const headerRow = rows[0];
            const headerCells = headerRow.split('|').filter(cell => cell.trim() !== '');

            console.log('Заголовки таблицы:', headerCells);

            let cultureColIndex = -1;
            let profitColIndex = -1;

            headerCells.forEach((header, index) => {
                const cleanHeader = header.trim().toLowerCase();
                if (cleanHeader.includes('культур') || cleanHeader.includes('crop') || cleanHeader.includes('название')) {
                    cultureColIndex = index;
                }
                if (cleanHeader.includes('прибыль') || cleanHeader.includes('profit') || cleanHeader.includes('руб')) {
                    profitColIndex = index;
                }
            });

            console.log(`Столбец культуры: ${cultureColIndex}, Столбец прибыли: ${profitColIndex}`);

            if (cultureColIndex === -1 || profitColIndex === -1) {
                console.log('Не найдены нужные столбцы, пропускаем таблицу');
                return;
            }

            for (let i = 1; i < rows.length; i++) {
                const row = rows[i];
                const cells = row.split('|').filter(cell => cell.trim() !== '');

                if (cells.length > Math.max(cultureColIndex, profitColIndex)) {
                    const cropName = cells[cultureColIndex].trim();
                    const profitCell = cells[profitColIndex].trim();

                    console.log(`Строка ${i}: Культура="${cropName}", Прибыль="${profitCell}"`);

                    if (isValidCropName(cropName)) {
                        const profitValue = extractProfitValue(profitCell);

                        if (profitValue !== null && profitValue > 1000 && profitValue < 10000000) {
                            result.crops.push({
                                crop: cropName,
                                profit: profitValue,
                                source: `table_${tableIndex + 1}`,
                                rowIndex: i
                            });
                            console.log(`✓ Добавлена культура: ${cropName} - ${profitValue} руб/га`);
                        } else {
                            console.log(`✗ Невалидная прибыль: ${profitValue} для культуры "${cropName}"`);
                        }
                    } else {
                        console.log(`✗ Невалидное название культуры: "${cropName}"`);
                    }
                }
            }
        });

        if (result.crops.length > 0) {
            const uniqueCrops = {};
            result.crops.forEach(crop => {
                if (!uniqueCrops[crop.crop] || uniqueCrops[crop.crop].profit < crop.profit) {
                    uniqueCrops[crop.crop] = crop;
                }
            });

            result.crops = Object.values(uniqueCrops);

            result.crops.sort((a, b) => b.profit - a.profit);

            result.bestCrop = result.crops[0].crop;
            result.profitPerHa = result.crops[0].profit;

            console.log('Финальные результаты:', {
                bestCrop: result.bestCrop,
                profitPerHa: result.profitPerHa,
                totalCrops: result.crops.length,
                crops: result.crops.map(c => ({ crop: c.crop, profit: c.profit }))
            });
        } else {
            console.warn('Не найдено валидных культур в таблицах');
        }

        return result;
    }

    function isValidCropName(cropName) {
        if (!cropName || cropName.length < 2 || cropName.length > 50) {
            return false;
        }

        const invalidPatterns = [
            /^\d+$/, // только цифры
            /^сам[а-я]*/i, // начинается с "сам..."
            /^выгодн[а-я]*/i, // начинается с "выгодн..."
            /^культур[а-я]*/i, // начинается с "культур..."
            /^прибыль/i, // начинается с "прибыль"
            /^руб/i, // начинается с "руб"
            /^итог/i, // начинается с "итог"
            /^всег/i, // начинается с "всег"
            /^средн/i // начинается с "средн"
        ];

        for (let pattern of invalidPatterns) {
            if (pattern.test(cropName.toLowerCase())) {
                return false;
            }
        }

        const validCrops = [
            'пшеница', 'ячмень', 'рожь', 'овёс', 'кукуруза', 'рис', 'гречиха',
            'подсолнечник', 'рапс', 'соя', 'горох', 'фасоль', 'чечевица',
            'картофель', 'свекла', 'морковь', 'капуста', 'томат', 'огурец',
            'лён', 'конопля', 'хлопок', 'сахарная свекла', 'кормовые', 'свёкла'
        ];

        const lowerCropName = cropName.toLowerCase();
        return validCrops.some(validCrop => lowerCropName.includes(validCrop));
    }

    function extractProfitValue(profitCell) {
        const cleanValue = profitCell.replace(/[^\d\s,]/g, '').replace(/,/g, '').trim();
        const numberMatch = cleanValue.match(/(\d[\d\s]*)/);

        if (numberMatch) {
            const numberString = numberMatch[1].replace(/\s/g, '');
            const profit = parseInt(numberString);

            return isNaN(profit) ? null : profit;
        }

        return null;
    }

    function saveCalculationToDatabase(fieldId, profitData, area, aiResponse) {
        const totalProfit = profitData.profitPerHa * area;

        const calculationData = {
            crops: profitData.crops,
            area: area,
            timestamp: new Date().toISOString(),
            best_crop: profitData.bestCrop,
            profit_per_ha: profitData.profitPerHa,
            total_profit: totalProfit
        };

        fetch('<?= Url::to(['giga-chat/save-calculation']) ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-Token': '<?= Yii::$app->request->csrfToken ?>'
            },
            body: new URLSearchParams({
                'field_id': fieldId,
                'best_crop': profitData.bestCrop,
                'profit_per_ha': profitData.profitPerHa,
                'total_profit': totalProfit,
                'calculation_data': JSON.stringify(calculationData),
                'ai_response': aiResponse.substring(0, 1000),
                '_csrf': '<?= Yii::$app->request->csrfToken ?>'
            })
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Расчет успешно сохранен в базу данных');
                    showSaveNotification(profitData.bestCrop, totalProfit, profitData.crops.length);
                } else {
                    console.error('Ошибка сохранения:', data.error);
                    showErrorNotification('Ошибка сохранения данных: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Ошибка сохранения:', error);
                showErrorNotification('Ошибка сети: ' + error.message);
            });
    }

    function showErrorNotification(message) {
        const notification = document.createElement('div');
        notification.className = 'alert alert-danger alert-dismissible fade show';
        notification.innerHTML = `
        <i class="bi bi-exclamation-triangle me-2"></i>
        <strong>Ошибка!</strong> ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

        const aiResponse = document.getElementById('ai-response');
        if (aiResponse) {
            aiResponse.prepend(notification);
        }
    }

    function showSaveNotification(bestCrop, totalProfit, cropsCount) {
        const notification = document.createElement('div');
        notification.className = 'alert alert-success alert-dismissible fade show';
        notification.innerHTML = `
        <i class="bi bi-database-check me-2"></i>
        <strong>Данные сохранены!</strong><br>
        • Рекомендуемая культура: <strong>${bestCrop}</strong><br>
        • Прогнозируемая прибыль: <strong>${totalProfit.toLocaleString()} руб</strong><br>
        • Проанализировано культур: <strong>${cropsCount}</strong>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

        const aiResponse = document.getElementById('ai-response');
        if (aiResponse) {
            aiResponse.prepend(notification);
        }
    }
</script>