<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "product_prices".
 *
 * @property int $id
 * @property string $district Район
 * @property string $product Продукт
 * @property string $price Цена руб/т
 * @property string|null $notes Примечание
 * @property int $created_at Дата создания
 * @property int $updated_at Дата обновления
 */
class ProductPrices extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%product_prices}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['district', 'product', 'price'], 'required'],
            [['price'], 'number', 'min' => 0],
            [['notes'], 'string'],
            [['created_at', 'updated_at'], 'integer'],
            [['district', 'product'], 'string', 'max' => 100],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'district' => 'Район',
            'product' => 'Продукт',
            'price' => 'Цена (руб/т)',
            'notes' => 'Примечание',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'timestamp' => [
                'class' => \yii\behaviors\TimestampBehavior::className(),
                'createdAtAttribute' => 'created_at',
                'updatedAtAttribute' => 'updated_at',
            ],
        ];
    }

    /**
     * @param string $district
     * @param string $product
     * @return float|null
     */
    public static function getPrice($district, $product)
    {
        $model = self::find()
            ->where(['district' => $district, 'product' => $product])
            ->one();

        return $model ? (float)$model->price : null;
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getPricesByDistrict($district)
    {
        return self::find()
            ->where(['district' => $district])
            ->indexBy('product')
            ->all();
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getProductsForDistrict($district)
    {
        return self::find()
            ->select(['product', 'price'])
            ->where(['district' => $district])
            ->indexBy('product')
            ->column();
    }

    /**
     * @return array
     */
    public static function getAllProducts()
    {
        return self::find()
            ->select('product')
            ->distinct()
            ->orderBy('product')
            ->column();
    }

    /**
     * @return array
     */
    public static function getDistrictsList()
    {
        return self::find()
            ->select('district')
            ->distinct()
            ->indexBy('district')
            ->column();
    }

    /**
     * @param string $district
     * @return \yii\db\ActiveQuery
     */
    public static function findByDistrict($district)
    {
        return self::find()->where(['district' => $district]);
    }

    /**
     * @return string
     */
    public function getFormattedPrice()
    {
        return Yii::$app->formatter->asDecimal($this->price, 0) . ' руб/т';
    }
}