$(document).ready(function () {
    let isSending = false; // Флаг для отслеживания отправки

    // Отключаем стандартное поведение ActiveForm
    $('#giga-chat-form').on('beforeSubmit', function (e) {
        e.preventDefault();
        if (!isSending) {
            sendMessage();
        }
        return false; // Предотвращаем стандартную отправку формы
    });

    // Кнопка очистки
    $('#clear-btn').on('click', function () {
        $('#message-input').val('');
        $('#response-container').hide();
        $('#error-container').hide();
    });

    // Обработка нажатия Enter (только Enter, без Ctrl)
    $('#message-input').on('keydown', function (e) {
        if (e.keyCode === 13 && !e.shiftKey) {
            e.preventDefault();
            if (!isSending) {
                sendMessage();
            }
        }
    });

    // Функция отправки сообщения
    function sendMessage() {
        if (isSending) return; // Защита от повторной отправки

        let message = $('#message-input').val().trim();
        let submitBtn = $('#submit-btn');
        let loading = $('#loading');
        let errorContainer = $('#error-container');
        let errorMessage = $('#error-message');

        // Валидация на клиенте
        if (!message) {
            errorMessage.text('Пожалуйста, введите сообщение');
            errorContainer.show();
            return false;
        }

        // Устанавливаем флаг отправки
        isSending = true;

        // Скрываем предыдущие ошибки
        errorContainer.hide();

        // Показываем индикатор загрузки
        submitBtn.prop('disabled', true);
        loading.show();

        // Добавляем сообщение пользователя в историю сразу
        addUserMessageToHistory(message);

        // Отправляем AJAX запрос
        $.ajax({
            url: '/giga-chat/send-message',
            type: 'POST',
            data: {
                message: message
            },
            dataType: 'json',
            success: function (response) {
                console.log('Response received:', response); // Debug
                if (response.success) {
                    // Добавляем ответ AI в историю
                    addAiMessageToHistory(response.response);

                    // Очищаем поле ввода
                    $('#message-input').val('');
                } else {
                    errorMessage.text(escapeHtml(response.error));
                    errorContainer.show();
                    addAiMessageToHistory('Ошибка: ' + response.error, true);
                }
            },
            error: function (xhr, status, error) {
                console.log('AJAX error:', error); // Debug
                let errorText = 'Ошибка сети';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorText = xhr.responseJSON.error;
                }
                errorMessage.text(escapeHtml(errorText));
                errorContainer.show();
                addAiMessageToHistory('Ошибка сети: ' + errorText, true);
            },
            complete: function () {
                // Сбрасываем флаг и восстанавливаем кнопку
                isSending = false;
                submitBtn.prop('disabled', false);
                loading.hide();
            }
        });
    }

    // Добавление сообщения пользователя в историю
    function addUserMessageToHistory(message) {
        let chatHistory = $('#chat-history');
        let messageHtml = '<div class="message user-message">' +
            '<strong>Вы:</strong>' +
            '<div class="message-content">' + escapeHtml(message).replace(/\n/g, '<br>') + '</div>' +
            '<small class="text-muted">' + new Date().toLocaleString() + '</small>' +
            '</div>';

        chatHistory.append(messageHtml);
        scrollToBottom();
    }

    // Добавление ответа AI в историю
    function addAiMessageToHistory(response, isError = false) {
        let chatHistory = $('#chat-history');
        let messageHtml = '<div class="message ai-message ' + (isError ? 'error-message' : '') + '">' +
            '<strong>GigaChat:</strong>' +
            '<div class="message-content">' + formatResponse(response) + '</div>' +
            '<small class="text-muted">' + new Date().toLocaleString() + '</small>' +
            '</div>';

        chatHistory.append(messageHtml);
        scrollToBottom();
    }

    // Прокрутка к нижней части чата
    function scrollToBottom() {
        let chatHistory = $('#chat-history');
        chatHistory.scrollTop(chatHistory[0].scrollHeight);
    }

    // Функция для форматирования ответа с экранированием
    function formatResponse(text) {
        if (!text) return 'Нет ответа от сервера';

        // Сначала экранируем весь HTML
        text = escapeHtml(text);

        // Затем заменяем переносы строк на HTML теги
        text = text.replace(/\n/g, '<br>');

        // Форматируем markdown-like синтаксис
        text = text.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
        text = text.replace(/\*(.*?)\*/g, '<em>$1</em>');
        text = text.replace(/_(.*?)_/g, '<em>$1</em>');

        return text;
    }

    // Функция для экранирования HTML
    function escapeHtml(unsafe) {
        if (!unsafe) return '';

        // Используем нативный метод для надежного экранирования
        const div = document.createElement('div');
        div.textContent = unsafe;
        return div.innerHTML;
    }

    // Загрузка истории при смене сессии
    function loadSessionHistory(sessionId) {
        $.ajax({
            url: '/giga-chat/get-history',
            type: 'POST',
            data: {session_id: sessionId},
            dataType: 'json',
            success: function (response) {
                if (response.success) {
                    displayHistory(response.history);
                    $('#current-session').text('Сессия: ' + response.sessionId.substring(0, 8) + '...');
                    $('#session-id').val(response.sessionId);
                }
            }
        });
    }

    // Отображение истории в чате
    function displayHistory(history) {
        let historyContainer = $('#chat-history');
        historyContainer.empty();

        history.forEach(function (message) {
            let userMsg = '<div class="message user-message">' +
                '<strong>Вы:</strong>' +
                '<div class="message-content">' + escapeHtml(message.user_message).replace(/\n/g, '<br>') + '</div>' +
                '<small class="text-muted">' + message.created_at + '</small>' +
                '</div>';

            let aiMsg = '<div class="message ai-message ' + (message.is_error ? 'error-message' : '') + '">' +
                '<strong>GigaChat:</strong>' +
                '<div class="message-content">' + formatResponse(message.ai_response) + '</div>' +
                '<small class="text-muted">' + message.created_at + '</small>' +
                '</div>';

            historyContainer.append(userMsg + aiMsg);
        });

        scrollToBottom();
    }

    // Обработчики для сессий (делегированная обработка)
    $(document).on('click', '.session-item', function () {
        loadSessionHistory($(this).data('session-id'));
    });

    $(document).on('click', '#new-session-btn', function (e) {
        e.preventDefault();
        $.ajax({
            url: '/giga-chat/new-session',
            type: 'POST',
            dataType: 'json',
            success: function (response) {
                if (response.success) {
                    loadSessionHistory(response.sessionId);
                }
            }
        });
    });

    // Прокрутка к нижней части при загрузке
    setTimeout(scrollToBottom, 100);
});